<?php
/**
 * Plugin Name: SteadFast API Integration
 * Description: SteadFast API integration with WooCommerce. Sends orders to SteadFast, saves response to 'shipments' table.
 * Version: 1.0.0
 * Author: steadfastit
 * Text Domain: steadfast-api
 * License: GPLv2
 */

defined( 'ABSPATH' ) || exit;

class STDF_Steadfast_API {

    public function __construct() {
        add_action('admin_menu', array($this, 'register_admin_menu'));
        add_action('admin_init', array($this, 'register_admin_settings'));
        add_action('wp_ajax_stdf_send_order', array($this, 'send_order_to_steadfast'));
    }

    /**
     * Register plugin settings page in WordPress admin menu
     */
    public function register_admin_menu() {
        add_menu_page(
            'SteadFast API', 
            'SteadFast API', 
            'manage_options', 
            'steadfast-api-settings', 
            array($this, 'settings_page_html'), 
            'dashicons-admin-generic'
        );
    }

    /**
     * Register settings fields for API Key, Secret Key, Auth Token
     */
    public function register_admin_settings() {
        register_setting('steadfast_api_settings', 'stdf_api_key');
        register_setting('steadfast_api_settings', 'stdf_api_secret');
        register_setting('steadfast_api_settings', 'stdf_auth_token');

        add_settings_section('stdf_api_section', 'SteadFast API Credentials', null, 'steadfast-api-settings');

        add_settings_field('stdf_api_key', 'API Key', array($this, 'render_field'), 'steadfast-api-settings', 'stdf_api_section', array('label_for' => 'stdf_api_key'));
        add_settings_field('stdf_api_secret', 'Secret Key', array($this, 'render_field'), 'steadfast-api-settings', 'stdf_api_section', array('label_for' => 'stdf_api_secret'));
        add_settings_field('stdf_auth_token', 'Auth Token (Bearer)', array($this, 'render_field'), 'steadfast-api-settings', 'stdf_api_section', array('label_for' => 'stdf_auth_token'));
    }

    /**
     * Settings field renderer
     */
    public function render_field($args) {
        $value = esc_attr(get_option($args['label_for']));
        echo "<input type='text' name='{$args['label_for']}' id='{$args['label_for']}' value='{$value}' class='regular-text'>";
    }

    /**
     * Settings Page HTML
     */
    public function settings_page_html() {
        ?>
        <div class="wrap">
            <h1>SteadFast API Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('steadfast_api_settings');
                do_settings_sections('steadfast-api-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Send order to SteadFast API and save to shipments table
     * AJAX handler: expects order_id in POST
     */
    public function send_order_to_steadfast() {
        global $wpdb;

        // Get API credentials from settings
        $api_key = get_option('stdf_api_key');
        $api_secret = get_option('stdf_api_secret');
        $auth_token = get_option('stdf_auth_token');

        // Get order_id from Ajax
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        if (!$order_id) {
            wp_send_json_error('Order ID missing');
        }

        // Example: Get order data (for WooCommerce)
        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }

        // Prepare API request data (customize as per SteadFast API required format)
        $request_payload = array(
            'recipient_name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
            'recipient_phone' => $order->get_billing_phone(),
            'recipient_address' => $order->get_billing_address_1(),
            'order_id' => $order_id,
            // Add more order info as required by SteadFast API...
        );

        // Prepare API call
        $api_url = 'https://portal.packzy.com/api/v1/create_shipment'; // Change to actual endpoint if needed

        $args = array(
            'headers' => array(
                'Content-Type'  => 'application/json',
                'api-key'       => $api_key,
                'secret-key'    => $api_secret,
                'Authorization' => 'Bearer ' . $auth_token,
            ),
            'body' => json_encode($request_payload),
            'timeout' => 30,
        );

        $response = wp_remote_post($api_url, $args);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);

        // Handle response and insert into 'shipments' table
        $status = isset($response_data['status']) ? $response_data['status'] : 'failed';
        $courier_order_id = isset($response_data['courier_order_id']) ? $response_data['courier_order_id'] : '';
        $tracking_number = isset($response_data['tracking_number']) ? $response_data['tracking_number'] : '';
        $pickup_requested_at = isset($response_data['pickup_requested_at']) ? $response_data['pickup_requested_at'] : '';
        $last_update = current_time('mysql');
        $created_at = current_time('mysql');

        // Save to shipments table
        $wpdb->insert(
            $wpdb->prefix . 'shipments',
            array(
                'order_id'          => $order_id,
                'courier_name'      => 'SteadFast',
                'courier_order_id'  => $courier_order_id,
                'tracking_number'   => $tracking_number,
                'status'            => $status,
                'pickup_requested_at'=> $pickup_requested_at,
                'last_update'       => $last_update,
                'request_payload'   => json_encode($request_payload),
                'response_payload'  => $response_body,
                'created_at'        => $created_at,
            ),
            array(
                '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s'
            )
        );

        if ($status == '200' || $status == 'success') {
            wp_send_json_success('Shipment created and saved!');
        } else {
            wp_send_json_error('API error: '. print_r($response_data, true));
        }
    }
}

new STDF_Steadfast_API();